/*
 * main.c -- the bare scull char module
 *
 * Copyright (C) 2001 Alessandro Rubini and Jonathan Corbet
 * Copyright (C) 2001 O'Reilly & Associates
 *
 * The source code in this file can be freely used, adapted,
 * and redistributed in source or binary form, so long as an
 * acknowledgment appears in derived source files.  The citation
 * should list that the code comes from the book "Linux Device
 * Drivers" by Alessandro Rubini and Jonathan Corbet, published
 * by O'Reilly & Associates.   No warranty is attached;
 * we cannot take responsibility for errors or fitness for use.
 *
 */

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

#if PROPERTY >= 2
#define USE_AUX
#endif

#if PROPERTY >= 4
#define MODEL_MALLOC
#endif

// C functions that are modeled directly
int *malloc(int size);

#define MODEL_MALLOC
#ifdef MODEL_MALLOC
int malloced[1000];

int stuck(){
LOOP: goto LOOP;
}

/* This is an amusing implementation of malloc. It picks a nondeterministic
   address then checks to see if "size" consecutive locations are available
   at this address, and aborts if not */

int *malloc(int size){
        int i,t;
        t = ac_malloc();
	
	for(i = 0; i < size; i++){
	  if (malloced[t+i]==1)
	    stuck();
	  malloced[t+i] = 1;
	}
        return t;
}

#endif

void memset(void *ptr, int val, int length);

// declarations of model functions
int sel(int mem, int p);
int ptr(int addr, int field);
int upd(int mem, int addr, int val);
int updflatrange(int mem, int lo, int hi, int v);
int updbothranges(int mem, int loAddr, int hiAddr, int loIndex, int hiIndex, int v);
int canReach(int mem, int field, int src, int dest);
int canReachWO(int mem, int field, int src, int dest, int avoid);
int noCycles(int mem, int field, int ptr);
int disjoint(int mem, int field, int p, int q);
int ptr_field(int ptr);
int ptr_addr(int ptr);
int metaVar(int whatever);

// need to somehow say this is a constant symbolic value, not a global..
// let's try 'const'
int const dead;
int const live;
//int const state;

// more or less true globals
int mem;      // for arrays
int meta;     // for arrays
int *tag;
int objsize;
int objct;
             
#define BLAST_ERROR

// experiment; problem is this wants the global invariant to
// hold, but I just want to model it simply.. modified vcgen.cc
// to do so, so this pre/post isn't used now
void kfree(void *freedPtr){
  if(tag[(int)freedPtr] != 0)
    tag[(int)freedPtr] = 0;
  else BLAST_ERROR;
}



// ------------------ from the kernel headers -----------------
// unknown source
typedef int loff_t;
typedef int ssize_t;
typedef int size_t;
typedef int filldir_t;
typedef int kdev_t;
typedef int pid_t;


// linux/kernel.h
#define printk printf
void printf(char const *fmt, ...);
#define	KERN_EMERG	"<0>"	/* system is unusable			*/
#define	KERN_ALERT	"<1>"	/* action must be taken immediately	*/
#define	KERN_CRIT	"<2>"	/* critical conditions			*/
#define	KERN_ERR	"<3>"	/* error conditions			*/
#define	KERN_WARNING	"<4>"	/* warning conditions			*/
#define	KERN_NOTICE	"<5>"	/* normal but significant condition	*/
#define	KERN_INFO	"<6>"	/* informational			*/
#define	KERN_DEBUG	"<7>"	/* debug-level messages			*/


// linux/fs.h
/*
 * NOTE:
 * read, write, poll, fsync, readv, writev can be called
 *   without the big kernel lock held in all filesystems.
 */
struct file;
struct poll_table_struct;
struct inode;
struct vm_area_struct;

typedef struct file_operations {
//	struct module *owner;
	loff_t (*llseek) (struct file *, loff_t, int);
	ssize_t (*read) (struct file *, char *, size_t, loff_t *);
	ssize_t (*write) (struct file *, const char *, size_t, loff_t *);
	int (*readdir) (struct file *, void *, filldir_t);
	unsigned int (*poll) (struct file *, struct poll_table_struct *);
	int (*ioctl) (struct inode *, struct file *, unsigned int, unsigned long);
	int (*mmap) (struct file *, struct vm_area_struct *);
	int (*open) (struct inode *, struct file *);
	int (*flush) (struct file *);
	int (*release) (struct inode *, struct file *);
//  	int (*fsync) (struct file *, struct dentry *, int datasync);
//  	int (*fasync) (int, struct file *, int);
//  	int (*lock) (struct file *, int, struct file_lock *);
//  	ssize_t (*readv) (struct file *, const struct iovec *, unsigned long, loff_t *);
//  	ssize_t (*writev) (struct file *, const struct iovec *, unsigned long, loff_t *);
//  	ssize_t (*sendpage) (struct file *, struct page *, int, size_t, loff_t *, int);
//  	unsigned long (*get_unmapped_area)(struct file *, unsigned long, unsigned long, unsigned long, unsigned long);
} file_operations;


typedef int ModuleId;
typedef enum FileStateEnum { FS_CLOSED, FS_OPEN } FileState;


// linux/fs.h
typedef struct file {
//  	struct list_head	f_list;
//  	struct dentry		*f_dentry;
//  	struct vfsmount         *f_vfsmnt;
	struct file_operations	*f_op;
//  	atomic_t		f_count;
  	unsigned int 		f_flags;
//  	mode_t			f_mode;
  	loff_t			f_pos;
//  	unsigned long 		f_reada, f_ramax, f_raend, f_ralen, f_rawin;
//  	struct fown_struct	f_owner;
//  	unsigned int		f_uid, f_gid;
//  	int			f_error;

//  	unsigned long		f_version;

//  	/* needed for tty driver, and maybe others */
  	void			*private_data;

//  	/* preallocated helper kiobuf to speedup O_DIRECT */
//  	struct kiobuf		*f_iobuf;
//  	long			f_iobuf_lock;

        // annotation (ghost) variables
        ModuleId                id;
        FileState               state;
} file;


// linux/fs.h
typedef struct inode {
//  	struct list_head	i_hash;
//  	struct list_head	i_list;
//  	struct list_head	i_dentry;

//  	struct list_head	i_dirty_buffers;
//  	struct list_head	i_dirty_data_buffers;

//  	unsigned long		i_ino;
//  	atomic_t		i_count;
//  	kdev_t			i_dev;
//  	umode_t			i_mode;
//  	nlink_t			i_nlink;
//  	uid_t			i_uid;
//  	gid_t			i_gid;
  	kdev_t			i_rdev;
//  	loff_t			i_size;
//  	time_t			i_atime;
//  	time_t			i_mtime;
//  	time_t			i_ctime;
//  	unsigned int		i_blkbits;
//  	unsigned long		i_blksize;
//  	unsigned long		i_blocks;
//  	unsigned long		i_version;
//  	struct semaphore	i_sem;
//  	struct semaphore	i_zombie;
//  	struct inode_operations	*i_op;
//  	struct file_operations	*i_fop;	/* former ->i_op->default_file_ops */
//  	struct super_block	*i_sb;
//  	wait_queue_head_t	i_wait;
//  	struct file_lock	*i_flock;
//  	struct address_space	*i_mapping;
//  	struct address_space	i_data;
//  	struct dquot		*i_dquot[MAXQUOTAS];
//  	/* These three should probably be a union */
//  	struct list_head	i_devices;
//  	struct pipe_inode_info	*i_pipe;
//  	struct block_device	*i_bdev;
//  	struct char_device	*i_cdev;

//  	unsigned long		i_dnotify_mask; /* Directory notify events */
//  	struct dnotify_struct	*i_dnotify; /* for directory notifications */

//  	unsigned long		i_state;

//  	unsigned int		i_flags;
//  	unsigned char		i_sock;

//  	atomic_t		i_writecount;
//  	unsigned int		i_attr_flags;
//  	__u32			i_generation;
//  	union {
//  		struct minix_inode_info		minix_i;
//  		struct ext2_inode_info		ext2_i;
//  		struct ext3_inode_info		ext3_i;
//  		struct hpfs_inode_info		hpfs_i;
//  		struct ntfs_inode_info		ntfs_i;
//  		struct msdos_inode_info		msdos_i;
//  		struct umsdos_inode_info	umsdos_i;
//  		struct iso_inode_info		isofs_i;
//  		struct nfs_inode_info		nfs_i;
//  		struct sysv_inode_info		sysv_i;
//  		struct affs_inode_info		affs_i;
//  		struct ufs_inode_info		ufs_i;
//  		struct efs_inode_info		efs_i;
//  		struct romfs_inode_info		romfs_i;
//  		struct shmem_inode_info		shmem_i;
//  		struct coda_inode_info		coda_i;
//  		struct smb_inode_info		smbfs_i;
//  		struct hfs_inode_info		hfs_i;
//  		struct adfs_inode_info		adfs_i;
//  		struct qnx4_inode_info		qnx4_i;
//  		struct reiserfs_inode_info	reiserfs_i;
//  		struct bfs_inode_info		bfs_i;
//  		struct udf_inode_info		udf_i;
//  		struct ncp_inode_info		ncpfs_i;
//  		struct proc_inode_info		proc_i;
//  		struct socket			socket_i;
//  		struct usbdev_inode_info        usbdev_i;
//  		struct jffs2_inode_info		jffs2_i;
//  		void				*generic_ip;
//  	} u;
} inode;


// linux/fs.h
int register_chrdev(unsigned int, const char *, struct file_operations *);
int unregister_chrdev(unsigned int, const char *);


// linux/sched.h
typedef struct task_struct {
//  	/*
//  	 * offsets of these are hardcoded elsewhere - touch with care
//  	 */
//  	volatile long state;	/* -1 unrunnable, 0 runnable, >0 stopped */
//  	unsigned long flags;	/* per process flags, defined below */
//  	int sigpending;
//  	mm_segment_t addr_limit;	/* thread address space:
//  					 	0-0xBFFFFFFF for user-thead
//  						0-0xFFFFFFFF for kernel-thread
//  					 */
//  	struct exec_domain *exec_domain;
//  	volatile long need_resched;
//  	unsigned long ptrace;

//  	int lock_depth;		/* Lock depth */

//  /*
//   * offset 32 begins here on 32-bit platforms. We keep
//   * all fields in a single cacheline that are needed for
//   * the goodness() loop in schedule().
//   */
//  	long counter;
//  	long nice;
//  	unsigned long policy;
//  	struct mm_struct *mm;
//  	int processor;
//  	/*
//  	 * cpus_runnable is ~0 if the process is not running on any
//  	 * CPU. It's (1 << cpu) if it's running on a CPU. This mask
//  	 * is updated under the runqueue lock.
//  	 *
//  	 * To determine whether a process might run on a CPU, this
//  	 * mask is AND-ed with cpus_allowed.
//  	 */
//  	unsigned long cpus_runnable, cpus_allowed;
//  	/*
//  	 * (only the 'next' pointer fits into the cacheline, but
//  	 * that's just fine.)
//  	 */
//  	struct list_head run_list;
//  	unsigned long sleep_time;

//  	struct task_struct *next_task, *prev_task;
//  	struct mm_struct *active_mm;
//  	struct list_head local_pages;
//  	unsigned int allocation_order, nr_local_pages;

//  /* task state */
//  	struct linux_binfmt *binfmt;
//  	int exit_code, exit_signal;
//  	int pdeath_signal;  /*  The signal sent when the parent dies  */
//  	/* ??? */
//  	unsigned long personality;
//  	int did_exec:1;
  	pid_t pid;
//  	pid_t pgrp;
//  	pid_t tty_old_pgrp;
//  	pid_t session;
//  	pid_t tgid;
//  	/* boolean value for session group leader */
//  	int leader;
//  	/* 
//  	 * pointers to (original) parent process, youngest child, younger sibling,
//  	 * older sibling, respectively.  (p->father can be replaced with 
//  	 * p->p_pptr->pid)
//  	 */
//  	struct task_struct *p_opptr, *p_pptr, *p_cptr, *p_ysptr, *p_osptr;
//  	struct list_head thread_group;

//  	/* PID hash table linkage. */
//  	struct task_struct *pidhash_next;
//  	struct task_struct **pidhash_pprev;

//  	wait_queue_head_t wait_chldexit;	/* for wait4() */
//  	struct completion *vfork_done;		/* for vfork() */
//  	unsigned long rt_priority;
//  	unsigned long it_real_value, it_prof_value, it_virt_value;
//  	unsigned long it_real_incr, it_prof_incr, it_virt_incr;
//  	struct timer_list real_timer;
//  	struct tms times;
//  	unsigned long start_time;
//  	long per_cpu_utime[NR_CPUS], per_cpu_stime[NR_CPUS];
//  /* mm fault and swap info: this can arguably be seen as either mm-specific or thread-specific */
//  	unsigned long min_flt, maj_flt, nswap, cmin_flt, cmaj_flt, cnswap;
//  	int swappable:1;
//  /* process credentials */
//  	uid_t uid,euid,suid,fsuid;
//  	gid_t gid,egid,sgid,fsgid;
//  	int ngroups;
//  	gid_t	groups[NGROUPS];
//  	kernel_cap_t   cap_effective, cap_inheritable, cap_permitted;
//  	int keep_capabilities:1;
//  	struct user_struct *user;
//  /* limits */
//  	struct rlimit rlim[RLIM_NLIMITS];
//  	unsigned short used_math;
//  	char comm[16];
//  /* file system info */
//  	int link_count, total_link_count;
//  	struct tty_struct *tty; /* NULL if no tty */
//  	unsigned int locks; /* How many file locks are being held */
//  /* ipc stuff */
//  	struct sem_undo *semundo;
//  	struct sem_queue *semsleeping;
//  /* CPU-specific state of this task */
//  	struct thread_struct thread;
//  /* filesystem information */
//  	struct fs_struct *fs;
//  /* open file information */
//  	struct files_struct *files;
//  /* signal handlers */
//  	spinlock_t sigmask_lock;	/* Protects signal and blocked */
//  	struct signal_struct *sig;

//  	sigset_t blocked;
//  	struct sigpending pending;

//  	unsigned long sas_ss_sp;
//  	size_t sas_ss_size;
//  	int (*notifier)(void *priv);
//  	void *notifier_data;
//  	sigset_t *notifier_mask;
	
//  /* Thread group tracking */
//     	u32 parent_exec_id;
//     	u32 self_exec_id;
//  /* Protection of (de-)allocation: mm, files, fs, tty */
//  	spinlock_t alloc_lock;

//  /* journalling filesystem info */
//  	void *journal_info;
} task_struct;
struct task_struct *current;    // current process


// asm/errno.h
#define	ENOMEM		12	/* Out of memory */
#define	EFAULT		14	/* Bad address */
#define	EBUSY		16	/* Device or resource busy */
#define	ENODEV		19	/* No such device */
#define	EINVAL		22	/* Invalid argument */
#define	EUCLEAN		117	/* Structure needs cleaning */
#define ERESTARTSYS     513     /* makee upee */
#define ENOTTY          514
#define EPERM           515

// asm/fcntl.h
#define O_ACCMODE	   0003
#define O_RDONLY	     00
#define O_WRONLY	     01
#define O_RDWR		     02
#define O_NONBLOCK	  04000


// asm/uaccess.h

unsigned long copy_to_user(void *to, const void *from, unsigned long n){
}

unsigned long copy_from_user(void *to, const void *from, unsigned long n){
}


#define __copy_from_user copy_from_user
int access_ok(int type, void *addr, int size);
#define VERIFY_READ 0
#define VERIFY_WRITE 1
#define get_user(var, ptr) (var = *(ptr), 0)
#define __get_user(var, ptr) (var = *(ptr), 0)
#define put_user(var, ptr) (*(ptr) = var, 0)


// linux/module.h
#define MODULE_PARM(var, flags)
#define MODULE_PARM_DESC(var, desc)
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT


// asm/ioctl.h
/* ioctl command encoding: 32 bits total, command in lower 16 bits,
 * size of the parameter structure in the lower 14 bits of the
 * upper 16 bits.
 * Encoding the size of the parameter structure in the ioctl request
 * is useful for catching programs compiled with old versions
 * and to avoid overwriting user space outside the user buffer area.
 * The highest 2 bits are reserved for indicating the ``access mode''.
 * NOTE: This limits the max parameter size to 16kB -1 !
 */

/*
 * The following is for compatibility across the various Linux
 * platforms.  The i386 ioctl numbering scheme doesn't really enforce
 * a type field.  De facto, however, the top 8 bits of the lower 16
 * bits are indeed used as a type field, so we might just as well make
 * this explicit here.  Please be sure to use the decoding macros
 * below from now on.
 */
#define _IOC_NRBITS	8
#define _IOC_TYPEBITS	8
#define _IOC_SIZEBITS	14
#define _IOC_DIRBITS	2

#define _IOC_NRMASK	((1 << _IOC_NRBITS)-1)
#define _IOC_TYPEMASK	((1 << _IOC_TYPEBITS)-1)
#define _IOC_SIZEMASK	((1 << _IOC_SIZEBITS)-1)
#define _IOC_DIRMASK	((1 << _IOC_DIRBITS)-1)

#define _IOC_NRSHIFT	0
#define _IOC_TYPESHIFT	(_IOC_NRSHIFT+_IOC_NRBITS)
#define _IOC_SIZESHIFT	(_IOC_TYPESHIFT+_IOC_TYPEBITS)
#define _IOC_DIRSHIFT	(_IOC_SIZESHIFT+_IOC_SIZEBITS)

/*
 * Direction bits.
 */
#define _IOC_NONE	0U
#define _IOC_WRITE	1U
#define _IOC_READ	2U

#define _IOC(dir,type,nr,size) 	(((dir)  << _IOC_DIRSHIFT) | 	 ((type) << _IOC_TYPESHIFT) | 	 ((nr)   << _IOC_NRSHIFT) | 	 ((size) << _IOC_SIZESHIFT))

/* used to create numbers */
#define _IO(type,nr)		_IOC(_IOC_NONE,(type),(nr),0)
#define _IOR(type,nr,size)	_IOC(_IOC_READ,(type),(nr),sizeof(size))
#define _IOW(type,nr,size)	_IOC(_IOC_WRITE,(type),(nr),sizeof(size))
#define _IOWR(type,nr,size)	_IOC(_IOC_READ|_IOC_WRITE,(type),(nr),sizeof(size))

/* used to decode ioctl numbers.. */
#define _IOC_DIR(nr)		(((nr) >> _IOC_DIRSHIFT) & _IOC_DIRMASK)
#define _IOC_TYPE(nr)		(((nr) >> _IOC_TYPESHIFT) & _IOC_TYPEMASK)
#define _IOC_NR(nr)		(((nr) >> _IOC_NRSHIFT) & _IOC_NRMASK)
#define _IOC_SIZE(nr)		(((nr) >> _IOC_SIZESHIFT) & _IOC_SIZEMASK)

/* ...and for the drivers/sound files... */

#define IOC_IN		(_IOC_WRITE << _IOC_DIRSHIFT)
#define IOC_OUT		(_IOC_READ << _IOC_DIRSHIFT)
#define IOC_INOUT	((_IOC_WRITE|_IOC_READ) << _IOC_DIRSHIFT)
#define IOCSIZE_MASK	(_IOC_SIZEMASK << _IOC_SIZESHIFT)
#define IOCSIZE_SHIFT	(_IOC_SIZESHIFT)


// unknown
#define MINOR(dev) (dev & 0xFF)
#define EXPORT_NO_SYMBOLS 


// ------------------ from mykmalloc.h --------------------
#define MYKMALLOCPRINTK(size) malloc(size)
#define mykfree(p) kfree(p)
#define mykfreesz(p,sz) kfree(p)
#define GFP_KERNEL 0
#define kmalloc(a,b) malloc(a)

// -------------------- from kutil.h -------------------
// return type of functions which return 0 or positive for
// success, and negative errno values for failure; other types
// that often have these semantics include ssize_t and loff_t
typedef int Err;

// integer as boolean
typedef int bool;
//enum { false=0, true=1 };
#define false 0
#define true 1


// probably the most ubiquitous macro...
#ifndef min
  #define min(a,b) ((a)<(b) ? (a) : (b))
#endif
  

// printk that includes the process id of the associated user process
#define printkpid(kernLevel, fmt, args...) printk(kernLevel "[%d] " fmt, current->pid , ## args);      

  
// standard debugging flags; standard flags will be capital
// letters, and module-specific ones will be lowercase
// (mnemonic for all letters: SYLBOG)
enum {
  // this end: generally useful
  DF_LEAKS = 'L',               // report memory leaks at unload time
  DF_OPEN_RELEASE = 'O',        // calls to open and release
  DF_SYSCALL = 'Y',             // system calls other than open and release
  DF_SIGNAL = 'G',              // signal activity
  DF_SLEEP = 'S',               // sleep/wakeup
  DF_BADIOCTL = 'B',            // print when an unknown ioctl is issued
  // this end: for debugging
};


#define ERRCALL(call)    err = call;            if (err < 0) {           return err;          }
    
  
  // NDEBUG branch
  #define debugFlagIsSet(flag) false
  #define dprintk(flag, fmt, args...) ((void)0);
  #define dprintkpid(flag, fmt, args...) ((void)0);

  #define CHECK(condition, codeIfFalse)           if (!(condition)) {                             return codeIfFalse;                         }


struct semaphore {};

/*
 * scull.h -- definitions for the char module
 *
 * Copyright (C) 2001 Alessandro Rubini and Jonathan Corbet
 * Copyright (C) 2001 O'Reilly & Associates
 *
 * The source code in this file can be freely used, adapted,
 * and redistributed in source or binary form, so long as an
 * acknowledgment appears in derived source files.  The citation
 * should list that the code comes from the book "Linux Device
 * Drivers" by Alessandro Rubini and Jonathan Corbet, published
 * by O'Reilly & Associates.   No warranty is attached;
 * we cannot take responsibility for errors or fitness for use.
 *
 * $Id: scull.c,v 1.1 2005/01/28 02:10:09 mcmillan Exp $
 */

#ifndef _SCULL_H_
#define _SCULL_H_

#if 0
#include <linux/ioctl.h> /* needed for the _IOW etc stuff used later */
#endif 

/* version dependencies have been confined to a separate file */
#if 0
#include "sysdep.h"
#endif

/*
 * Macros to help debugging
 */

#undef PDEBUG             /* undef it, just in case */
#ifdef SCULL_DEBUG
#  ifdef __KERNEL__
     /* This one if debugging is on, and kernel space */
#    define PDEBUG(fmt, args...) printk( KERN_DEBUG "scull: " fmt, ## args)
#  else
     /* This one for user space */
#    define PDEBUG(fmt, args...) fprintf(stderr, fmt, ## args)
#  endif
#else
#  define PDEBUG(fmt, args...) /* not debugging: nothing */
#endif

#undef PDEBUGG
#define PDEBUGG(fmt, args...) /* nothing: it's a placeholder */

#ifndef SCULL_MAJOR
#define SCULL_MAJOR 0   /* dynamic major by default */
#endif

#ifndef SCULL_NR_DEVS
#define SCULL_NR_DEVS 4    /* scull0 through scull3 */
#endif

#ifndef SCULL_P_NR_DEVS
#define SCULL_P_NR_DEVS 4  /* scullpipe0 through scullpipe3 */
#endif

/*
 * The bare device is a variable-length region of memory.
 * Use a linked list of indirect blocks.
 *
 * "Scull_Dev->data" points to an array of pointers, each
 * pointer refers to a memory area of SCULL_QUANTUM bytes.
 *
 * The array (quantum-set) is SCULL_QSET long.
 */
#ifndef SCULL_QUANTUM
#define SCULL_QUANTUM 4000
#endif

#ifndef SCULL_QSET
#define SCULL_QSET    1000
#endif

/*
 * The pipe device is a simple circular buffer. Here its default size
 */
#ifndef SCULL_P_BUFFER
#define SCULL_P_BUFFER 4000
#endif

#ifdef CONFIG_DEVFS_FS /* only if enabled, to avoid errors in 2.0 */
#if 0
#include <linux/devfs_fs_kernel.h>
#endif
#else
  typedef void * devfs_handle_t;  /* avoid #ifdef inside the structure */
#endif

/*
 * This is somehow a hack: avoid ifdefs in the cleanup code by declaring
 * an empty procedure as a placeholder for devfs_unregister. This is
 * only done *unless* <linux/devfs_fs_kernel.h> was included, as that
 * header already implements placeholder for all the devfs functions
 */
/*............................................... degin-tag devfs-ifdef */
#ifndef DEVFS_FL_DEFAULT
extern inline void devfs_unregister(devfs_handle_t de) {}
#endif

extern devfs_handle_t scull_devfs_dir;


typedef struct Scull_Dev {
   void **data;
   struct Scull_Dev *next;   /* next listitem */
   int quantum;              /* the current quantum size */
   int qset;                 /* the current array size */
   unsigned long size;
   devfs_handle_t handle;    /* only used if devfs is there */
   unsigned int access_key;  /* used by sculluid and scullpriv */
   struct semaphore sem;     /* mutual exclusion semaphore     */
#define USE_AUX
#ifdef USE_AUX
  struct Scull_Dev *prev;    /* inverse of "next" */
#endif
} Scull_Dev;

/*
 * Split minors in two parts
 */
//#define TYPE(dev)   (MINOR(dev) >> 4)  /* high nibble */
//#define NUM(dev)    (MINOR(dev) & 0xf) /* low  nibble */

/* can't handle the above because of bit vector arith */
#define TYPE(dev)   0  /* high nibble */
#define NUM(dev)    dev /* low  nibble */

/*
 * Different minors behave differently, so let's use multiple fops
 */

extern struct file_operations scull_fops;        /* simplest: global */
extern struct file_operations scull_priv_fops;   /* private region   */
extern struct file_operations scull_pipe_fops;   /* circular buffer  */
extern struct file_operations scull_sngl_fops;   /* single open      */
extern struct file_operations scull_user_fops;   /* single process   */
extern struct file_operations scull_wusr_fops;   /* single user      */

/*
 * The different configurable parameters
 */
extern int scull_major;     /* main.c */
extern int scull_nr_devs;
extern int scull_quantum;
extern int scull_qset;

extern int scull_p_nr_devs;    /* pipe.c */
extern int scull_p_buffer;


/*
 * Prototypes for shared functions
 */

int     scull_p_init(void);
void    scull_p_cleanup(void);
int     scull_access_init(void);
void    scull_access_cleanup(void);

int     scull_trim(Scull_Dev *dev);

ssize_t scull_read (struct file *filp, char *buf, size_t count,
                    loff_t *f_pos);
ssize_t scull_write (struct file *filp, const char *buf, size_t count,
                     loff_t *f_pos);
loff_t  scull_llseek (struct file *filp, loff_t off, int whence);
int     scull_ioctl (struct inode *inode, struct file *filp,
                     unsigned int cmd, unsigned long arg);


/*
 * Ioctl definitions
 */

/* Use 'k' as magic number */
#define SCULL_IOC_MAGIC  'k'

#define SCULL_IOCRESET    _IO(SCULL_IOC_MAGIC, 0)

/*
 * S means "Set" through a ptr,
 * T means "Tell" directly with the argument value
 * G means "Get": reply by setting through a pointer
 * Q means "Query": response is on the return value
 * X means "eXchange": G and S atomically
 * H means "sHift": T and Q atomically
 */
#define SCULL_IOCSQUANTUM _IOW(SCULL_IOC_MAGIC,  1, scull_quantum)
#define SCULL_IOCSQSET    _IOW(SCULL_IOC_MAGIC,  2, scull_qset)
#define SCULL_IOCTQUANTUM _IO(SCULL_IOC_MAGIC,   3)
#define SCULL_IOCTQSET    _IO(SCULL_IOC_MAGIC,   4)
#define SCULL_IOCGQUANTUM _IOR(SCULL_IOC_MAGIC,  5, scull_quantum)
#define SCULL_IOCGQSET    _IOR(SCULL_IOC_MAGIC,  6, scull_qset)
#define SCULL_IOCQQUANTUM _IO(SCULL_IOC_MAGIC,   7)
#define SCULL_IOCQQSET    _IO(SCULL_IOC_MAGIC,   8)
#define SCULL_IOCXQUANTUM _IOWR(SCULL_IOC_MAGIC, 9, scull_quantum)
#define SCULL_IOCXQSET    _IOWR(SCULL_IOC_MAGIC,10, scull_qset)
#define SCULL_IOCHQUANTUM _IO(SCULL_IOC_MAGIC,  11)
#define SCULL_IOCHQSET    _IO(SCULL_IOC_MAGIC,  12)

/*
 * The other entities only have "Tell" and "Query", because they're
 * not printed in the book, and there's no need to have all six.
 * (The previous stuff was only there to show different ways to do it.
 */
#define SCULL_P_IOCTSIZE _IO(SCULL_IOC_MAGIC,   13)
#define SCULL_P_IOCQSIZE _IO(SCULL_IOC_MAGIC,   14)
/* ... more to come */
#define SCULL_IOCHARDRESET _IO(SCULL_IOC_MAGIC, 15) /* debugging tool */

#define SCULL_IOC_MAXNR 15

#endif /* _SCULL_H_ */

#define NULL 0
#define CAP_SYS_ADMIN 0
#define capable(x) 0

enum ModuleState { MS_UNLOADED, MS_LOADED } module_state = MS_UNLOADED;


/*
 * I don't use static symbols here, because we export no symbols
 */

int scull_major =   SCULL_MAJOR;
/* int scull_nr_devs = SCULL_NR_DEVS; */   /* number of bare scull devices */
int scull_quantum = SCULL_QUANTUM;
int scull_qset =    SCULL_QSET;

/* MODULE_PARM(scull_major,"i");
MODULE_PARM(scull_nr_devs,"i");
MODULE_PARM(scull_quantum,"i");
MODULE_PARM(scull_qset,"i");
 MODULE_AUTHOR("Alessandro Rubini"); */

Scull_Dev *scull_devices; /* allocated in scull_init_module */

/*
 * Different minors behave differently, so let's use multiple fops
 */


struct file_operations *scull_fop_array[]={
    &scull_fops,      /* type 0 */
    &scull_priv_fops, /* type 1 */
    &scull_pipe_fops, /* type 2 */
    &scull_sngl_fops, /* type 3 */
    &scull_user_fops, /* type 4 */
    &scull_wusr_fops  /* type 5 */
};
#define SCULL_MAX_TYPE 5


int scull_trim(Scull_Dev *dev)
{
    Scull_Dev *next, *dptr;
    int qset = dev->qset;   /* "dev" is not-null */
    int i;

    for (dptr = dev; dptr; dptr = next) { /* all the list items */
        if (dptr->data) {
            for (i = 0; i < qset; i++)
                if (dptr->data[i])
                    kfree(dptr->data[i]);
            kfree(dptr->data);
            dptr->data=NULL;
        }
        next=dptr->next;
        if (dptr != dev) kfree(dptr); /* all of them but the first */
    }
    dev->size = 0;
    dev->quantum = scull_quantum;
    dev->qset = scull_qset;
    dev->next = NULL;
    return 0;
}
#ifdef SCULL_DEBUG /* use proc only if debugging */
/*
 * The proc filesystem: function to read and entry
 */

int scull_read_procmem(char *buf, char **start, off_t offset,
                   int count, int *eof, void *data)
{
    int i, j, len = 0;
    int limit = count - 80; /* Don't print more than this */

    for (i = 0; i < scull_nr_devs && len <= limit; i++) {
        Scull_Dev *d = &scull_devices[i];
        if (down_interruptible(&d->sem))
                return -ERESTARTSYS;
        len += sprintf(buf+len,"\nDevice %i: qset %i, q %i, sz %li\n",
                       i, d->qset, d->quantum, d->size);
        for (; d && len <= limit; d = d->next) { /* scan the list */
            len += sprintf(buf+len, "  item at %p, qset at %p\n", d, d->data);
            if (d->data && !d->next) /* dump only the last item - save space */
                for (j = 0; j < d->qset; j++) {
                    if (d->data[j])
                        len += sprintf(buf+len,"    % 4i: %8p\n",j,d->data[j]);
                }
        }
        up(&scull_devices[i].sem);
    }
    *eof = 1;
    return len;
}

#ifdef USE_PROC_REGISTER

static int scull_get_info(char *buf, char **start, off_t offset,
                int len, int unused)
{
    int eof = 0;
    return scull_read_procmem (buf, start, offset, len, &eof, NULL);
}

struct proc_dir_entry scull_proc_entry = {
        namelen:    8,
        name:       "scullmem",
        mode:       S_IFREG | S_IRUGO,
        nlink:      1,
        get_info:   scull_get_info,
};

static void scull_create_proc()
{
    proc_register_dynamic(&proc_root, &scull_proc_entry);
}

static void scull_remove_proc()
{
    proc_unregister(&proc_root, scull_proc_entry.low_ino);
}

#else  /* no USE_PROC_REGISTER - modern world */

static void scull_create_proc()
{
    create_proc_read_entry("scullmem", 0 /* default mode */,
                           NULL /* parent dir */, scull_read_procmem,
                           NULL /* client data */);
}

static void scull_remove_proc()
{
    /* no problem if it was not registered */
    remove_proc_entry("scullmem", NULL /* parent dir */);
}


#endif /* USE_PROC_REGISTER */





#endif /* SCULL_DEBUG */





/*
 * Open and close
 */


/* In scull_open, the fop_array is used according to TYPE(dev) */
int scull_open(struct inode *inode, struct file *filp)
{
    Scull_Dev *dev; /* device information */
    int num = NUM(inode->i_rdev);
    int type = TYPE(inode->i_rdev);

    if(num < 0) while(1); /* bit vector arithmetic fact we can't prove */

    /*
     * the type and num values are only valid if we are not using devfs.
     * However, since we use them to retrieve the device pointer, we
     * don't need them with devfs as filp->private_data is already
     * initialized
     */

    /*
     * If private data is not valid, we are not using devfs
     * so use the type (from minor nr.) to select a new f_op
     */


#define NO_FUNCTION_POINTERS
#ifndef NO_FUNCTION_POINTERS
    if (!filp->private_data && type) {
        if (type > SCULL_MAX_TYPE) return -ENODEV;
        filp->f_op = scull_fop_array[type];
        return filp->f_op->open(inode, filp); /* dispatch to specific open */
    }
#endif

    /* type 0, check the device number (unless private_data valid) */
    dev = (Scull_Dev *)filp->private_data;
    if (!dev) {
        if (num >= scull_nr_devs) return -ENODEV;
        dev = &scull_devices[num];
        filp->private_data = dev; /* for other methods */
    }

    MOD_INC_USE_COUNT;  /* Before we maybe sleep */
    /* now trim to 0 the length of the device if open was write-only */
    if ( (filp->f_flags & O_ACCMODE) == O_WRONLY) {
        if (down_interruptible(&dev->sem)) {
            MOD_DEC_USE_COUNT;
            return -ERESTARTSYS;
        }
        scull_trim(dev); /* ignore errors */
        up(&dev->sem);
    }

    return 0;          /* success */
}

int scull_release(struct inode *inode, struct file *filp)
{
    MOD_DEC_USE_COUNT;
    return 0;
}

#define sizeof(x) 1

/*
 * Follow the list 
 */
Scull_Dev *scull_follow(Scull_Dev *dev, int n)
{
    while (n--) {
        if (!dev->next) {
            dev->next = kmalloc(sizeof(Scull_Dev), GFP_KERNEL);
            memset(dev->next, 0, sizeof(Scull_Dev));
#ifdef USE_AUX
	    dev->next->prev = dev;
#endif
        }
        dev = dev->next;
        continue;
    }
    return dev;
}

/*
 * Data management: read and write
 */

ssize_t scull_read(struct file *filp, char *buf, size_t count,
                loff_t *f_pos)
{
    Scull_Dev *dev = filp->private_data; /* the first listitem */
    Scull_Dev *dptr;
    int quantum = dev->quantum;
    int qset = dev->qset;
    int itemsize = quantum * qset; /* how many bytes in the listitem */
    int item, s_pos, q_pos, rest;
    ssize_t ret = 0;

    if (down_interruptible(&dev->sem))
            return -ERESTARTSYS;
    if (*f_pos >= dev->size)
        goto out;
    if (*f_pos + count > dev->size)
        count = dev->size - *f_pos;
    /* find listitem, qset index, and offset in the quantum */
    item = (long)*f_pos / itemsize;
    rest = (long)*f_pos % itemsize;
    s_pos = rest / quantum; q_pos = rest % quantum;

    /* follow the list up to the right position (defined elsewhere) */
    dptr = scull_follow(dev, item);

    if (!dptr->data)
        goto out; /* don't fill holes */
    if (!dptr->data[s_pos])
        goto out;
    /* read only up to the end of this quantum */
    if (count > quantum - q_pos)
        count = quantum - q_pos;

    if (copy_to_user(buf, dptr->data[s_pos]+q_pos, count)) {
        ret = -EFAULT;
	goto out;
    }
    *f_pos += count;
    ret = count;

 out:
    up(&dev->sem);
    return ret;
}

ssize_t scull_write(struct file *filp, const char *buf, size_t count,
                loff_t *f_pos)
{
    Scull_Dev *dev = filp->private_data;
    Scull_Dev *dptr;
    int quantum = dev->quantum;
    int qset = dev->qset;
    int itemsize = quantum * qset;
    int item, s_pos, q_pos, rest;
    ssize_t ret = -ENOMEM; /* value used in "goto out" statements */

    if (down_interruptible(&dev->sem))
            return -ERESTARTSYS;

    /* find listitem, qset index and offset in the quantum */
    item = (long)*f_pos / itemsize;
    rest = (long)*f_pos % itemsize;
    s_pos = rest / quantum; q_pos = rest % quantum;

    /* follow the list up to the right position */
    dptr = scull_follow(dev, item);
    if (!dptr->data) {
        dptr->data = kmalloc(qset * sizeof(char *), GFP_KERNEL);
        if (!dptr->data)
            goto out;
        memset(dptr->data, 0, qset * sizeof(char *));
    }
    if (!dptr->data[s_pos]) {
        dptr->data[s_pos] = kmalloc(quantum, GFP_KERNEL);
        if (!dptr->data[s_pos])
            goto out;
    }
    /* write only up to the end of this quantum */
    if (count > quantum - q_pos)
        count = quantum - q_pos;

    if (copy_from_user(dptr->data[s_pos]+q_pos, buf, count)) {
        ret = -EFAULT;
	goto out;
    }
    *f_pos += count;
    ret = count;

    /* update the size */
    if (dev->size < *f_pos)
        dev-> size = *f_pos;

  out:
    up(&dev->sem);
    return ret;
}

/*
 * The ioctl() implementation
 *
 * This is done twice, once the 2.2 way, followed by the 2.0 way.  One
 * would not normally do things in this manner, but we wanted to illustrate
 * both ways...
 */

#ifndef LINUX_20

int scull_ioctl(struct inode *inode, struct file *filp,
                 unsigned int cmd, unsigned long arg)
{

    int err = 0, tmp;
    int ret = 0;
    
    /*
     * extract the type and number bitfields, and don't decode
     * wrong cmds: return ENOTTY (inappropriate ioctl) before access_ok()
     */
    if (_IOC_TYPE(cmd) != SCULL_IOC_MAGIC) return -ENOTTY;
    if (_IOC_NR(cmd) > SCULL_IOC_MAXNR) return -ENOTTY;

    /*
     * the direction is a bitmask, and VERIFY_WRITE catches R/W
     * transfers. `Type' is user-oriented, while
     * access_ok is kernel-oriented, so the concept of "read" and
     * "write" is reversed
     */
    if (_IOC_DIR(cmd) & _IOC_READ)
        err = !access_ok(VERIFY_WRITE, (void *)arg, _IOC_SIZE(cmd));
    else if (_IOC_DIR(cmd) & _IOC_WRITE)
        err =  !access_ok(VERIFY_READ, (void *)arg, _IOC_SIZE(cmd));
    if (err) return -EFAULT;

    switch(cmd) {

#ifdef SCULL_DEBUG
      case SCULL_IOCHARDRESET:
         /*
          * reset the counter to 1, to allow unloading in case
          * of problems. Use 1, not 0, because the invoking
          * process has the device open.
          */
         while (MOD_IN_USE)
             MOD_DEC_USE_COUNT;
         MOD_INC_USE_COUNT;
         /* don't break: fall through and reset things */
#endif /* SCULL_DEBUG */

      case SCULL_IOCRESET:
        scull_quantum = SCULL_QUANTUM;
        scull_qset = SCULL_QSET;
        break;
        
      case SCULL_IOCSQUANTUM: /* Set: arg points to the value */
        if (! capable (CAP_SYS_ADMIN))
            return -EPERM;
        ret = __get_user(scull_quantum, (int *)arg);
        break;

      case SCULL_IOCTQUANTUM: /* Tell: arg is the value */
        if (! capable (CAP_SYS_ADMIN))
            return -EPERM;
        scull_quantum = arg;
        break;

      case SCULL_IOCGQUANTUM: /* Get: arg is pointer to result */
        ret = __put_user(scull_quantum, (int *)arg);
        break;

      case SCULL_IOCQQUANTUM: /* Query: return it (it's positive) */
        return scull_quantum;

      case SCULL_IOCXQUANTUM: /* eXchange: use arg as pointer */
        if (! capable (CAP_SYS_ADMIN))
            return -EPERM;
        tmp = scull_quantum;
        ret = __get_user(scull_quantum, (int *)arg);
        if (ret == 0)
            ret = __put_user(tmp, (int *)arg);
        break;

      case SCULL_IOCHQUANTUM: /* sHift: like Tell + Query */
        if (! capable (CAP_SYS_ADMIN))
            return -EPERM;
        tmp = scull_quantum;
        scull_quantum = arg;
        return tmp;
        
      case SCULL_IOCSQSET:
        if (! capable (CAP_SYS_ADMIN))
            return -EPERM;
        ret = __get_user(scull_qset, (int *)arg);
        break;

      case SCULL_IOCTQSET:
        if (! capable (CAP_SYS_ADMIN))
            return -EPERM;
        scull_qset = arg;
        break;

      case SCULL_IOCGQSET:
        ret = __put_user(scull_qset, (int *)arg);
        break;

      case SCULL_IOCQQSET:
        return scull_qset;

      case SCULL_IOCXQSET:
        if (! capable (CAP_SYS_ADMIN))
            return -EPERM;
        tmp = scull_qset;
        ret = __get_user(scull_qset, (int *)arg);
        if (ret == 0)
            ret = put_user(tmp, (int *)arg);
        break;

      case SCULL_IOCHQSET:
        if (! capable (CAP_SYS_ADMIN))
            return -EPERM;
        tmp = scull_qset;
        scull_qset = arg;
        return tmp;

        /*
         * The following two change the buffer size for scullpipe.
         * The scullpipe device uses this same ioctl method, just to
         * write less code. Actually, it's the same driver, isn't it?
         */

      case SCULL_P_IOCTSIZE:
        scull_p_buffer = arg;
        break;

      case SCULL_P_IOCQSIZE:
        return scull_p_buffer;


      default:  /* redundant, as cmd was checked against MAXNR */
        return -ENOTTY;
    }
    return ret;

}

#else  /* LINUX_20 */

int scull_ioctl(struct inode *inode, struct file *filp,
                 unsigned int cmd, unsigned long arg)
{

    int err = 0, tmp;
    
    /*
     * extract the type and number bitfields, and don't decode
     * wrong cmds: return ENOTTY before verify_area()
     */
    if (_IOC_TYPE(cmd) != SCULL_IOC_MAGIC) return -ENOTTY;
    if (_IOC_NR(cmd) > SCULL_IOC_MAXNR) return -ENOTTY;

    /*
     * the direction is a bitmask, and VERIFY_WRITE catches R/W
     * transfers. `Type' is user-oriented, while
     * verify_area is kernel-oriented, so the concept of "read" and
     * "write" is reversed
     */
    if (_IOC_DIR(cmd) & _IOC_READ)
        err = verify_area(VERIFY_WRITE, (void *)arg, _IOC_SIZE(cmd));
    else if (_IOC_DIR(cmd) & _IOC_WRITE)
        err =  verify_area(VERIFY_READ, (void *)arg, _IOC_SIZE(cmd));
    if (err) return err;

    switch(cmd) {

      case SCULL_IOCRESET:
        scull_quantum = SCULL_QUANTUM;
        scull_qset = SCULL_QSET;
        break;
        
      case SCULL_IOCSQUANTUM: /* Set: arg points to the value */
        scull_quantum = get_user((int *)arg);
        break;

      case SCULL_IOCTQUANTUM: /* Tell: arg is the value */
        scull_quantum = arg;
        break;

      case SCULL_IOCGQUANTUM: /* Get: arg is pointer to result */
        put_user(scull_quantum, (int *)arg);
        break;

      case SCULL_IOCQQUANTUM: /* Query: return it (it's positive) */
        return scull_quantum;

      case SCULL_IOCXQUANTUM: /* eXchange: use arg as pointer */
        tmp = scull_quantum;
        scull_quantum = get_user((int *)arg);
        put_user(tmp, (int *)arg);
        break;

      case SCULL_IOCHQUANTUM: /* sHift: like Tell + Query */
        tmp = scull_quantum;
        scull_quantum = arg;
        return tmp;
        
      case SCULL_IOCSQSET:
        scull_qset = get_user((int *)arg);
        break;

      case SCULL_IOCTQSET:
        scull_qset = arg;
        break;

      case SCULL_IOCGQSET:
        put_user(scull_qset, (int *)arg);
        break;

      case SCULL_IOCQQSET:
        return scull_qset;

      case SCULL_IOCXQSET:
        tmp = scull_qset;
        scull_qset = get_user((int *)arg);
        put_user(tmp, (int *)arg);
        break;

      case SCULL_IOCHQSET:
        tmp = scull_qset;
        scull_quantum = arg;
        return tmp;

        /*
         * The following two change the buffer size for scullpipe.
         * The scullpipe device uses this same ioctl method, just to
         * write less code. Actually, it's the same driver, isn't it?
         */

      case SCULL_P_IOCTSIZE:
        scull_p_buffer = arg;
        break;

      case SCULL_P_IOCQSIZE:
        return scull_p_buffer;


      default:  /* redundant, as cmd was checked against MAXNR */
        return -ENOTTY;
    }
    return 0;

}


#endif /* LINUX_20 */


/*
 * The "extended" operations -- only seek
 */

loff_t scull_llseek(struct file *filp, loff_t off, int whence)
{
    Scull_Dev *dev = filp->private_data;
    loff_t newpos;

    switch(whence) {
      case 0: /* SEEK_SET */
        newpos = off;
        break;

      case 1: /* SEEK_CUR */
        newpos = filp->f_pos + off;
        break;

      case 2: /* SEEK_END */
        newpos = dev->size + off;
        break;

      default: /* can't happen */
        return -EINVAL;
    }
    if (newpos<0) return -EINVAL;
    filp->f_pos = newpos;
    return newpos;
}


/*
 * The following wrappers are meant to make things work with 2.0 kernels
 */
#ifdef LINUX_20
int scull_lseek_20(struct inode *ino, struct file *f,
                off_t offset, int whence)
{
    return (int)scull_llseek(f, offset, whence);
}

int scull_read_20(struct inode *ino, struct file *f, char *buf, int count)
{
    return (int)scull_read(f, buf, count, &f->f_pos);
}

int scull_write_20(struct inode *ino, struct file *f, const char *b, int c)
{
    return (int)scull_write(f, b, c, &f->f_pos);
}

void scull_release_20(struct inode *ino, struct file *f)
{
    scull_release(ino, f);
}

/* Redefine "real" names to the 2.0 ones */
#define scull_llseek scull_lseek_20
#define scull_read scull_read_20
#define scull_write scull_write_20
#define scull_release scull_release_20
#define llseek lseek
#endif  /* LINUX_20 */

struct file_operations scull_fops = {
    llseek:     scull_llseek,
    read:       scull_read,
    write:      scull_write,
    ioctl:      scull_ioctl,
    open:       scull_open,
    release:    scull_release,
};

/*
 * Finally, the module stuff
 */

#ifdef CONFIG_DEVFS_FS
devfs_handle_t scull_devfs_dir;
static char devname[4];
#endif

/*
 * The cleanup function is used to handle initialization failures as well.
 * Thefore, it must be careful to work correctly even if some of the items
 * have not been initialized
 */
void scull_cleanup_module(void)
{
    int i;

#ifndef CONFIG_DEVFS_FS
    /* cleanup_module is never called if registering failed */
    unregister_chrdev(scull_major, "scull");
#endif

#ifdef SCULL_DEBUG /* use proc only if debugging */
    scull_remove_proc();
#endif
    if (scull_devices) {
        for (i=0; i<scull_nr_devs; i++) {
            scull_trim(scull_devices+i);
            /* the following line is only used for devfs */
            devfs_unregister(scull_devices[i].handle);
        }
        kfree(scull_devices);
    }

    /* and call the cleanup functions for friend devices */
    scull_p_cleanup();
    scull_access_cleanup();

    /* once again, only for devfs */
    devfs_unregister(scull_devfs_dir);

}


int scull_init_module(void)
{
    int result, i;

    SET_MODULE_OWNER(&scull_fops);
#ifdef CONFIG_DEVFS_FS
    /* If we have devfs, create /dev/scull to put files in there */
    scull_devfs_dir = devfs_mk_dir(NULL, "scull", NULL);
    if (!scull_devfs_dir) return -EBUSY; /* problem */

#else /* no devfs, do it the "classic" way  */    

    /*
     * Register your major, and accept a dynamic number. This is the
     * first thing to do, in order to avoid releasing other module's
     * fops in scull_cleanup_module()
     */
    result = register_chrdev(scull_major, "scull", &scull_fops);
    if (result < 0) {
        printk(KERN_WARNING "scull: can't get major %d\n",scull_major);
        return result;
    }
    if (scull_major == 0) scull_major = result; /* dynamic */

#endif /* CONFIG_DEVFS_FS */
    /* 
     * allocate the devices -- we can't have them static, as the number
     * can be specified at load time
     */
    scull_devices = kmalloc(scull_nr_devs * sizeof(Scull_Dev), GFP_KERNEL);
    if (!scull_devices) {
        result = -ENOMEM;
        goto fail;
    }
    memset(scull_devices, 0, scull_nr_devs * sizeof(Scull_Dev));
    for (i=0; i < scull_nr_devs; i++) {
        scull_devices[i].quantum = scull_quantum;
        scull_devices[i].qset = scull_qset;
        sema_init(&scull_devices[i].sem, 1);
#ifdef USE_AUX
	scull_devices[i].next = 0; /* because memset isn't modeled */
#endif
#ifdef CONFIG_DEVFS_FS
        sprintf(devname, "%i", i);
        devfs_register(scull_devfs_dir, devname,
                       DEVFS_FL_AUTO_DEVNUM,
                       0, 0, S_IFCHR | S_IRUGO | S_IWUGO,
                       &scull_fops,
                       scull_devices+i);
#endif  
    }

#if 0
    /* At this point call the init function for any friend device */
    if ( (result = scull_p_init()) )
        goto fail;
    if ( (result = scull_access_init()) )
        goto fail;
    /* ... */
#endif

#ifndef SCULL_DEBUG
    EXPORT_NO_SYMBOLS; /* otherwise, leave global symbols visible */
#endif

#ifdef SCULL_DEBUG /* only when debugging */
    scull_create_proc();
#endif

    module_state = MS_LOADED;
    return 0; /* succeed */

  fail:
    scull_cleanup_module();
    return result;
}


#if 0
module_init(scull_init_module);
module_exit(scull_cleanup_module);
#endif

error(){
 ERROR: goto ERROR;
}

main(){
  
#if PROPERTY == 0
  if(module_state == MS_UNLOADED)
    if(scull_init_module() >= 0)
      if(module_state != MS_LOADED){
	error();
      }
#endif

#if PROPERTY == 1
  int j;

  if(module_state == MS_UNLOADED)
    if(scull_init_module() >= 0)
      if(0 <= j && j < scull_nr_devs && scull_devices[j].quantum != scull_quantum)
	error();
  
#endif

#if PROPERTY == 2
  int j;

  if(module_state == MS_UNLOADED)
    if(scull_init_module() >= 0){
      if(0 <= j && j < scull_nr_devs){
	scull_follow(&scull_devices[j],1);
	if(scull_devices[j].next->prev != &scull_devices[j])
	  error();
      }
    }

#endif

#if PROPERTY == 3
  struct inode *inode;
  struct file *filp;

  if ((*(scull_devices+((*inode).i_rdev))).next == 0) skip();  /* just to introduce this lvalue */

  if(filp->private_data == 0 && inode->i_rdev >= 0 && module_state == MS_UNLOADED)
    if(scull_init_module() >= 0)
      if(scull_open(inode,filp) == 0)
	if(((Scull_Dev *)(filp->private_data))->next != 0)
	  error();
#endif

#if PROPERTY == 4
  int j;

  if(module_state == MS_UNLOADED)
    if(scull_init_module() >= 0){
      if(0 <= j && j < scull_nr_devs){
	scull_follow(&scull_devices[j],1);
	if(scull_devices[j].next->prev != &scull_devices[j])
	  error();
      }
    }

#endif

#if PROPERTY == 5
  int j;
  int n;

  if (malloced[(int)(scull_devices+j)] == 1) skip();  /* just to introduce this lvalue */
  if (malloced[(int)((*(scull_devices+j)).next)] == 1) skip();  /* just to introduce this lvalue */
  if(module_state == MS_UNLOADED)
    if(scull_init_module() >= 0){
      if(0 <= j && j < scull_nr_devs&& n >= 1){
	scull_follow(&scull_devices[j],n);
	if(scull_devices[j].next->prev != &scull_devices[j])
	  error();
      }
    }

#endif

#if PROPERTY == 6
  Scull_dev *dev;
  int j;
  int n;
  
  dev->next = 0;
  if(module_state == MS_UNLOADED)
    if(scull_init_module() >= 0){
      if(0 <= j && j < scull_nr_devs&& n >= 1){
	scull_follow(&scull_devices[j],n);
	if(dev->next != 0 && dev->next->prev != dev)
	  error();
      }
    }

#endif
}
